#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>   /* mkdir */
#include <sys/types.h>  /* mkdir */
#include <errno.h>

int main(int argc, char *argv[]) {
    if (argc != 3) {
        fprintf(stderr, "Usage: %s input.fasta mode\n", argv[0]);
        fprintf(stderr, "   mode = 0  -> one file per sequence in current directory\n");
        fprintf(stderr, "   mode = 1  -> one folder per sequence, file inside folder\n");
        return 1;
    }

    int mode = atoi(argv[2]);

    FILE *in = fopen(argv[1], "r");
    if (!in) {
        perror("Error opening input file");
        return 1;
    }

    char line[8192];   /* allow long headers */
    FILE *out = NULL;
    char filename[1024];
    int writing = 0;

    while (fgets(line, sizeof(line), in)) {
        if (line[0] == '>') {
            /* Close previous file */
            if (out) {
                /*fputc('\n', out);*/
                fclose(out);
                out = NULL;
            }

            /* Find [protein_id=...] */
            char *protein_ptr = strstr(line, "[protein_id=");
            if (!protein_ptr) {
                fprintf(stderr, "Warning: no protein_id found in header: %s", line);
                writing = 0;
                continue;
            }

            protein_ptr += strlen("[protein_id=");
            char *end = strchr(protein_ptr, ']');
            if (!end) {
                fprintf(stderr, "Warning: malformed protein_id field: %s", line);
                writing = 0;
                continue;
            }

            /* Extract protein_id */
            size_t len = end - protein_ptr;
            if (len >= sizeof(filename) - 1) len = sizeof(filename) - 2;
            strncpy(filename, protein_ptr, len);
            filename[len] = '\0';

            if (mode == 0) {
                /* Simple mode: just filename.txt */
                strcat(filename, ".txt");
                out = fopen(filename, "w");
            } else {
                /* Mode 1: create a directory with protein_id, put sequence.txt inside */
                if (mkdir(filename, 0777) != 0 && errno != EEXIST) {
                    perror("Error creating directory");
                    fclose(in);
                    return 1;
                }

                char fullpath[2048];
                snprintf(fullpath, sizeof(fullpath), "%s/%s.txt", filename, filename);
                out = fopen(fullpath, "w");
            }

            if (!out) {
                perror("Error creating output file");
                fclose(in);
                return 1;
            }
            writing = 1;
        } else if (writing) {
            /* Sequence line: remove newline characters */
            line[strcspn(line, "\r\n")] = '\0';
            fprintf(out, "%s", line);
        }
    }

    if (out) {
        /*fputc('\n', out);*/
        fclose(out);
    }

    fclose(in);
    return 0;
}
